<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       http://wpgenie.org
 * @since      1.0.0
 *
 * @package    Woocommerce_Simple_Storewide_Sale
 * @subpackage Woocommerce_Simple_Storewide_Sale/public
 */

/**
 * Simple_Storewide_Sale_For_Woocommerce_Public
 */
class Simple_Storewide_Sale_For_Woocommerce_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;


	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param string $plugin_name       The name of the plugin.
	 * @param string $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Wc_simple_storewide_sale_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Wc_simple_storewide_sale_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/wc-simple-storewide-sale-public.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Wc_simple_storewide_sale_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Wc_simple_storewide_sale_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/wc-simple-storewide-sale-public.js', array( 'jquery' ), $this->version, false );
	}

	/**
	 * Function that checks if we need to run storewide sale
	 *
	 * @since    1.0.0
	 */
	public function run_sales() {

		$sale_enable = get_option( 'wcsss_enable', 'no' );
		$discount    = get_option( 'wcsss_amount', '' );
		if ( 'yes' === $sale_enable && ! empty( $discount ) && $this->is_sale_started() && ! $this->is_sale_ended() ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Functions check if sales are started
	 *
	 * @since    1.0.0
	 */
	public function is_sale_started() {

		$sale_from = get_option( 'wcsss_start', false );

		if ( isset( $sale_from ) && ! empty( $sale_from ) ) {
			$date1 = new DateTime( $sale_from );
			$date2 = new DateTime( current_time( 'mysql' ) );

			return ( $date1 < $date2 );
		} else {

			return false;
		}
	}


	/**
	 * Functions check if sales are ended
	 *
	 * @since    1.0.0
	 */
	public function is_sale_ended() {

		$sale_to = get_option( 'wcsss_end', false );

		if ( isset( $sale_to ) && ! empty( $sale_to ) ) {
			$date1 = new DateTime( $sale_to );
			$date2 = new DateTime( current_time( 'mysql' ) );

			return ( $date1 < $date2 );

		} else {

			return false;
		}
	}
	/**
	 * Functions that actualy changes price of products
	 *
	 * @param  string $price price.
	 * @param  object $product product.
	 * @param  object $parent_product parent product.
	 * @return string
	 */
	public function filterprice( $price, $product, $parent_product = null ) {

		global $sssfw;

		$newprice = 0;

		$product_regular_price = get_post_meta( $product->get_id(), '_regular_price', true );
		$product_sale_price    = get_post_meta( $product->get_id(), '_sale_price', true );
		$product_price         = get_post_meta( $product->get_id(), '_price', true );
		$is_product_on_sale    = $product_price === $product_sale_price;

		if ( sssfw_fs()->is_premium() ) {
			$wcsss_exclude_sale    = get_option( 'wcsss_exclude_sale', 'no' );
			if ( 'yes' === $wcsss_exclude_sale && $is_product_on_sale ) {
				return $price;
			}
		}

		if ( $price !== $product_price && ( current_filter() === 'woocommerce_product_get_price' || current_filter() === 'woocommerce_product_variation_get_price' ) ) {
			return $price;
		}

		if ( $price !== $product_sale_price && ( current_filter() === 'woocommerce_product_get_sale_price' || current_filter() === 'woocommerce_product_variation_get_sale_price' ) ) {
			return $price;
		}

		$orginal_product_id = method_exists( $product, 'get_id' ) ? $product->get_id() : $product->id;
		$discount           = floatval( get_option( 'wcsss_amount', '' ) );
		$type               = get_option( 'wcsss_type', '0' );

		if ( $parent_product ) {
			$product_id = method_exists( $parent_product, 'get_id' ) ? $parent_product->get_id() : $parent_product->id;
		} elseif ( $product->is_type( 'variation' ) && ! $parent_product ) {
			$product_id = method_exists( $product, 'get_parent_id' ) ? $product->get_parent_id() : '';
		} else {
			$product_id = method_exists( $product, 'get_id' ) ? $product->get_id() : $product->id;
		}

		$product_ids_on_sale = $sssfw->get_onsale_page_transient();

		if ( false === $product_ids_on_sale && is_array( $product_ids_on_sale ) ) {
			return $price;
		}
		if ( ! in_array( $product_id, $product_ids_on_sale ) ) {
			return $price;
		}
		if ( empty( $price ) ) {
			$price = get_post_meta( $orginal_product_id, '_price', true );
		}

		if ( $is_product_on_sale && get_option( 'wcsss_use_regular_price', 'no' ) === 'yes' ) {
			$price = $product_regular_price;
		}
		if ( '0' === $type && is_numeric( $price ) ) {
			$newprice = $price - ( $price * ( $discount / 100 ) );

		} elseif ( '1' === $type && is_numeric( $price ) ) {
			global $woocommerce_wpml;
			$newprice = $price - $this->wpml_convert_price( $discount );
		} else {
			do_action( 'wc_simple_storewide_sale_calculate_price_' . $type, $price, $product );
		}

		if ( $newprice > 0 && $newprice < $price ) {
			return $newprice;
		} else {
			return $price;
		}
	}

	/**
	 * Functions that shows notice on frontend
	 *
	 * @since    1.0.0
	 */
	public function sale_notice() {
		if ( sssfw_fs()->is_premium() ) {
			if ( $this->run_sales() ) {

				$wcsss_notice_enable = get_option( 'wcsss_notice_enable', 'no' );
				$wcsss_notice        = get_option( 'wcsss_notice', '' );

				if ( 'yes' === $wcsss_notice_enable && ! empty( $wcsss_notice ) ) {
					echo wp_kses_post( apply_filters( 'woocommerce_demo_store', '<p class="woocommerce-store-notice demo_store storewide_sales">' . $wcsss_notice . '</p>' ) );
				}
			} elseif ( ! $this->is_sale_started() && ! $this->is_sale_ended() ) {

				$wcsss_notice_enable = get_option( 'wcsss_before_notice_enable', 'no' );
				$wcsss_notice        = get_option( 'wcsss_notice_before', '' );

				if ( 'yes' === $wcsss_notice_enable && ! empty( $wcsss_notice ) ) {
					echo wp_kses_post( apply_filters( 'woocommerce_demo_store', '<p class="woocommerce-store-notice demo_store storewide_sales storewide_sales_before">' . $wcsss_notice . '</p>' ) );
				}
			}
		} elseif ( ! sssfw_fs()->is_premium() ) {
			if ( $this->run_sales() ) {

				$wcsss_notice_enable = get_option( 'wcsss_notice_enable', 'no' );
				$wcsss_notice        = get_option( 'wcsss_notice', '' );

				if ( 'yes' === $wcsss_notice_enable && ! empty( $wcsss_notice ) ) {
					echo wp_kses_post( apply_filters( 'woocommerce_demo_store', '<p class="woocommerce-store-notice demo_store storewide_sales">' . $wcsss_notice . '</p>' ) );
				}
			}
		}
	}
	/**
	 *  Get main product id for multilanguage purpose
	 *
	 * @param  mixed $id product id.
	 * @param  mixed $type type.
	 * @return string
	 */
	public function get_main_wpml_id( $id, $type ) {
		global $sitepress;

		if ( function_exists( 'icl_object_id' ) && function_exists( 'pll_default_language' ) ) { // Polylang with use of WPML compatibility mode.
			$id = icl_object_id( $id, $type, false, pll_default_language() );

		} elseif ( function_exists( 'icl_object_id' ) && method_exists( $sitepress, 'get_default_language' ) ) { // WPML.
			$id = icl_object_id( $id, $type, false, $sitepress->get_default_language() );

		}

		return $id;
	}

	/**
	 * Convert discoun currency
	 *
	 * @param  mixed $price price.
	 * @return string
	 */
	public function wpml_convert_price( $price ) {
		global $woocommerce_wpml;
		if ( function_exists( 'icl_object_id' ) && isset( $woocommerce_wpml ) && isset( $woocommerce_wpml->multi_currency->prices ) ) {
			$price = $woocommerce_wpml->multi_currency->prices->convert_price_amount( $price );
		}
			return $price;
	}

	/**
	 * Add discount hash key
	 *
	 * @param  mixed $hash hash.
	 * @param  mixed $product product.
	 * @return string
	 */
	public function add_discount_woocommerce_get_variation_prices_hash( $hash, $product ) {

		$hidden_hash = get_option( 'wcsss_hiddenhash', '' );
		$type        = get_option( 'wcsss_type', '0' );
		$hash[]      = get_option( 'wcsss_amount', '' ) . '-' . WC()->session->client_currency . '-' . $type . $hidden_hash . $hash[0];
		return $hash;
	}

	/**
	 * Add for OnSale Page for WooCommerce plugin compatibility
	 *
	 * @param  mixed $product_ids_on_sale product id on sale.
	 * @return array
	 */
	public function add_wc_onsale_page_product_ids_on_sale( $product_ids_on_sale ) {
		if ( $this->run_sales() ) {
			$added_ids_on_sale = get_transient( 'wc_onsale_page_products_onsale' );
			if ( false !== $added_ids_on_sale && is_array( $added_ids_on_sale ) ) {
				$product_ids_on_sale = array_merge( $product_ids_on_sale, $added_ids_on_sale );
			}
		}
		return $product_ids_on_sale;
	}
}
